/*----------------------------------------------------------------------------
 * Name:    ble process.c
 * Purpose:
 * Note(s):
 *----------------------------------------------------------------------------
 * This file is part of the uVision/ARM development tools.
 * This software may only be used under the terms of a valid, current,
 * end user licence from KEIL for a compatible version of KEIL software
 * development tools. Nothing else gives you the right to use this software.
 *
 * This software is supplied "AS IS" without warranties of any kind.
 *
 * Copyright (c) 2012 Keil - An ARM Company. All rights reserved.
 *----------------------------------------------------------------------------*/

#include		<string.h>
#include		<stdlib.h>
#include		"timer.h"
#include		"bc7701.h"
#include		"hci.h"
#include		"service.h"
#include		"bleprocess.h"

#if (_BDNAME_CFG_ENABLE_ == 1)
/* set BT device name */
uc8	BLE_DeviceName[] = {'B','M','7','7','0','1'};
#endif

#if (_ADV_DATA_CFG_ENABLE_ == 1)
/* set Advertising Data */
uc8	BLE_AdvData[] =
{
	/* flag */
	2,0x01,0x06,
//	8,0x16,0x1C,0x2A,0x01,0x23,0x00,0x00,0x00,
	/* Manufacturer Specific Data */
	11,0xFF,0xFF,0xFF,
	'B','e','s','t','C','o','m','m'
};
#endif

#if (_SCAN_DATA_CFG_ENABLE_ == 1)
/* Set Scan Response Data */
uc8	BLE_ScanData[] =
{
	/* complete List of 16-bit Service Class UUIDs */
	11,0x03,0x00,0x18,0x01,0x18,0x0A,0x18,0x0F,0x18,0xF0,0xFF
};
#endif

#if (_DEV_INF_CFG_ENABLE_ == 1)
uc8	BLE_DevInfSysID[] = { 0x37,0x37,0x30,0x31 };
uc8	BLE_DevInfModelNumber[] = {'B','M','7','7','0','1'};		//max 16 byte
uc8	BLE_DevInfSerialNumber[] = {'1','.','0','0'};
uc8	BLE_DevInfFirmwareRevs[] = {'1','.','0','0'};
uc8	BLE_DevInfHardwareRevs[] = {'1','.','0','0'};
uc8	BLE_DevInfSofewareRevs[] = {'1','.','0','0'};
uc8	BLE_DevInfManufacturer[] = {'B','e','s','t','C','o','m','m'};		//max 16 byte
uc8	BLE_DevInfIEEE11073[] = {0x00,0x00,0x00,0x00};
uc8	BLE_DevInfPnPID[] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00};
#endif

#if (_ADV_INTV_CFG_ENABLE_ == 1)
/* set Advertising interval */
const tADV_INTV	BLE_AdvInterval =
{
	(u16)(_ADV_INTV_MIN_VALUE_/0.625),(u16)(_ADV_INTV_MAX_VALUE_/0.625),0x07
};
#endif

tBLE_OPERATE BLE_Operate;

#define	EVTRSP_QUEUE_MAX			16
tEVENT_RESPONSE	*EventRspQueue[EVTRSP_QUEUE_MAX];
u8			uERQueueWr,uERQueueRd;
static u8	NextStepStatus;
#define IS_ERQUEUE_FULL(LEN)		(((uERQueueWr + LEN) >= EVTRSP_QUEUE_MAX) ? ((uERQueueWr + LEN - EVTRSP_QUEUE_MAX) == uERQueueRd) : ((uERQueueWr + LEN) == uERQueueRd))
#define IS_ERQUEUE_EMPTY()      	(uERQueueWr == uERQueueRd)

void *uReceiveIdx;
tEVENT_RESPONSE	*uRspEventIdx;

/*------------------------------------------------------------------------------
* Function Name : BLE_ParameterInitial
* Description   : BLE parmeter initial
* Input         :
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BLE_ParameterInitial(void)
{
	BLE_Operate.States = _BLE_START_;
	BLE_Operate.ConnectState = FALSE;
	//release malloc memory
	while(!IS_ERQUEUE_EMPTY())
	{
		free(EventRspQueue[uERQueueRd]);
		if(++uERQueueRd == EVTRSP_QUEUE_MAX) uERQueueRd = 0;
	}
}
/*------------------------------------------------------------------------------
* Function Name : BLE_ParameterConfigure
* Description   : BLE parmeter
* Input         :
*				- item : Configured items
* Output        : None
* Return        : None
* 				- TRUE : The configured item is successful
* 				- FALSE: The configured item is empty(indicating the end of the configuration)
------------------------------------------------------------------------------*/
void *BLE_ParameterConfigure(u8 *item)
{
	void *pret;

	switch(*item)
	{
		case 0:
#if (_CLOAD_CFG_ENABLE_ == 1)
			if((pret=BC7701_SetCrystalCload(_CLOAD_VALUE_)) != NULL) *item=1; //to next modify
			break;
#endif
		case 1:
#if (_TXPWR_CFG_ENABLE_ == 1)
			if((pret=BC7701_SetTxPower(_TX_POWER_VALUE_)) != NULL)	*item=2;	//to next modify
			break;
#endif
		case 2:
#if (_BDNAME_CFG_ENABLE_ == 1)
			if((pret=BC7701_SetDeviceName(sizeof(BLE_DeviceName),(u8 *)BLE_DeviceName)) != NULL)
				*item=3;					//to next modify
			break;
#endif
		case 3:
#if (_ADV_DATA_CFG_ENABLE_ == 1)
			if((pret=BC7701_SetAdvertisingData(UPDAE_AUTO_NAME,sizeof(BLE_AdvData),(u8 *)BLE_AdvData)) != NULL)
				*item=4;					//to next modify
			break;
#endif
		case 4:
#if (_SCAN_DATA_CFG_ENABLE_ == 1)
			if((pret=BC7701_SetScanResponseData(sizeof(BLE_ScanData),(u8 *)BLE_ScanData)) != NULL)
				*item=5;					//to next modify
			break;
#endif
		case 5:
#if (_ADV_INTV_CFG_ENABLE_ == 1)
			if((pret=BC7701_AdvertisingInterval(BLE_AdvInterval.min,BLE_AdvInterval.max,BLE_AdvInterval.map)) != NULL)
				*item=6;					//to next modify
			break;
#endif
#if (_DEV_INF_CFG_ENABLE_ == 1)
		case 6:
			if((pret=BC7701_SendBCIPackage(DevInf_SystemID,BCI_ServiceMessage,sizeof(BLE_DevInfSysID),(u8 *)BLE_DevInfSysID)) != NULL)
				*item=7;
			break;
		case 7:
			if((pret=BC7701_SendBCIPackage(ModelNumberString,BCI_ServiceMessage,sizeof(BLE_DevInfModelNumber),(u8 *)BLE_DevInfModelNumber)) != NULL)
				*item=8;
			break;
		case 8:
			if((pret=BC7701_SendBCIPackage(SerialNumberString,BCI_ServiceMessage,sizeof(BLE_DevInfSerialNumber),(u8 *)BLE_DevInfSerialNumber)) != NULL)
				*item=9;
			break;
		case 9:
			if((pret=BC7701_SendBCIPackage(FirmwareRevisionString,BCI_ServiceMessage,sizeof(BLE_DevInfFirmwareRevs),(u8 *)BLE_DevInfFirmwareRevs)) != NULL)
				*item=10;
			break;
		case 10:
			if((pret=BC7701_SendBCIPackage(HardwareRevisionString,BCI_ServiceMessage,sizeof(BLE_DevInfHardwareRevs),(u8 *)BLE_DevInfHardwareRevs)) != NULL)
				*item=11;
			break;
		case 11:
			if((pret=BC7701_SendBCIPackage(SoftwareRevisionString,BCI_ServiceMessage,sizeof(BLE_DevInfSofewareRevs),(u8 *)BLE_DevInfSofewareRevs)) != NULL)
				*item=12;
			break;
		case 12:
			if((pret=BC7701_SendBCIPackage(ManufacturerNameString,BCI_ServiceMessage,sizeof(BLE_DevInfManufacturer),(u8 *)BLE_DevInfManufacturer)) != NULL)
				*item=13;
			break;
		case 13:
			if((pret=BC7701_SendBCIPackage(IEEE11073_20601RegulatoryCertificationDataList,BCI_ServiceMessage,sizeof(BLE_DevInfIEEE11073),(u8 *)BLE_DevInfIEEE11073)) != NULL)
				*item=14;
			break;
		case 14:
			if((pret=BC7701_SendBCIPackage(DevInf_PnPID,BCI_ServiceMessage,sizeof(BLE_DevInfPnPID),(u8 *)BLE_DevInfPnPID)) != NULL)
				*item=15;
			break;
#endif
		default:
			*item = 0xFF;
			pret = NULL;
			break;
	}
	return(pret);
}
/*------------------------------------------------------------------------------
* Function Name : BLE_TimerProcess
* Description   : BLE timer counter(unit 1ms)
* Input         :
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BLE_TimerProcess(void)
{
	u8	idx;

   if(BLE_Operate.TimeCounter != 0) BLE_Operate.TimeCounter--;
	idx = uERQueueRd;
	while(idx != uERQueueWr)
	{
		if(EventRspQueue[idx]->flag == EVENT_RSP_UNDONE)
		{
			if(--EventRspQueue[idx]->TimeCounter == 0)
				EventRspQueue[idx]->flag = EVENT_RSP_TMOUT;
		}
		if(++idx >= EVTRSP_QUEUE_MAX) idx = 0;
	}
}
/*------------------------------------------------------------------------------
* Function Name : BLE_Transceive
* Description   : Transmit Package & recevie event response
* Input         :
*				- timeout : receive event response time
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
bool BLE_Transceive(void *package,u16 timeout)
{
	tEVENT_RESPONSE	*p;

	if(IS_ERQUEUE_FULL(1)) return FALSE;
	p = malloc(sizeof(tEVENT_RESPONSE));
	if(p == NULL) return FALSE;
	if(BC7701_TransmitPackage(package) == FALSE)
	{
		free(p);
		return(FALSE);
	}
	p->flag = EVENT_RSP_UNDONE;
	p->TimeCounter = timeout;
	switch(((u8 *)package)[0])
	{
		case HCI_CMD_PKG :
			p->RspType = HCI_EVENT_PKG;
			p->RspOpCode = ((tHCI_CMD_PKG *)package)->opcode;
			break;
		case BCI_CMD_PKG :
			p->RspType = BCI_EVENT_PKG;
			if(((tBCI_PACKAGE *)package)->opcode >= GenericAccess)
			{
				p->RspFlag = ((tBCI_PACKAGE *)package)->flag & ~EVTS_CODE_MASK;
			}
			else
			{
				p->RspFlag = 0x00;
			}
			p->RspOpCode = ((tBCI_PACKAGE *)package)->opcode;
			break;
	}
	EventRspQueue[uERQueueWr] = p;
	if(++uERQueueWr >= EVTRSP_QUEUE_MAX) uERQueueWr = 0;
	return(TRUE);
}
/*------------------------------------------------------------------------------
* Function Name : BLE_EventResponseCheck
* Description   : event response check
* Input         :
*				- pkgvalue : event package value flag
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BLE_EventRspQueueRemove(u8 idx)
{
	if(idx == uERQueueRd)
	{
		if(++uERQueueRd >= EVTRSP_QUEUE_MAX) uERQueueRd = 0;
	}
	else
	{
		while(idx != uERQueueWr)
		{
			if(idx < (EVTRSP_QUEUE_MAX-1))
				EventRspQueue[idx] = EventRspQueue[idx+1];
			else
				EventRspQueue[idx] = EventRspQueue[0];
			if(++idx >= EVTRSP_QUEUE_MAX) idx = 0;
		}
		if(uERQueueWr == 0) uERQueueWr = (EVTRSP_QUEUE_MAX-1);
		else uERQueueWr--;
	}
}
/*------------------------------------------------------------------------------
* Function Name : BLE_EventResponseCheck
* Description   : event response check
* Input         :
*				- pkgvalue : event package value flag
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
tEVENT_RESPONSE	*BLE_EventResponseCheck(void *package)
{
	u8	idx;
	tEVENT_RESPONSE	*revt = NULL,*p;

	idx = uERQueueRd;
	while(idx != uERQueueWr)
	{
		p = EventRspQueue[idx];
		if(((u8 *)package)[0] == p->RspType)
		{
			switch(((u8 *)package)[0])
			{
				case HCI_EVENT_PKG :
					if(((tHCI_EVENT_PKG *)package)->event == EVENT_CMD_COMPLETE)
					{
						if(*((u16 *)(((tHCI_EVENT_PKG *)package)->param)) == p->RspOpCode)
						{
							if(((tHCI_EVENT_PKG *)package)->param[2] == 0)		//check event status
								p->flag = EVENT_RSP_DONE;
							else
								p->flag = EVENT_RSP_ERROR;
							revt = p;
						}
					}
					break;
				case BCI_EVENT_PKG :
					if((((tBCI_PACKAGE *)package)->opcode == p->RspOpCode) &&
						((((tBCI_PACKAGE *)package)->flag & ~EVTS_CODE_MASK) == p->RspFlag))
					{
						if((((tBCI_PACKAGE *)package)->flag & EVTS_CODE_MASK) == EVTS_SUCCESS)
							p->flag = EVENT_RSP_DONE;
						else
							p->flag = EVENT_RSP_ERROR;
						revt = p;
					}
					break;
			}
		}
		if(revt != NULL)
		{
			BLE_EventRspQueueRemove(idx);
			return(revt);
		}
		else
		{
			if(++idx >= EVTRSP_QUEUE_MAX) idx = 0;
		}
	}
	return(revt);
}
/*------------------------------------------------------------------------------
* Function Name : BLE_TransceiveToEventRsp
* Description   :
* Input         :
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BLE_TransceiveToEventRsp(void *pack,u16 timeout,u8 next)
{
	BLE_Transceive(pack,timeout);
	BLE_Operate.States = _BLE_EVENT_RESPONSE_;
	NextStepStatus = next;
}
/*------------------------------------------------------------------------------
* Function Name : BLE_ErrorEventRsp
* Description   : Event Response time out error
* Input         :
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
uc8 HCI_TimeOutEventRsp[] =
{
	HCI_EVENT_PKG,
	EVENT_CMD_COMPLETE,
   4,
   1,
   0x00,0x00,0x04
};

uc8 BCI_TimeOutEventRsp[] =
{
	BCI_EVENT_PKG,
	3,
	EVTS_TIME_OUT,
	0x00,0x00,
	0x00
};

void *BLE_ErrorEventRsp(u8 idx)
{
	tEVENT_RESPONSE	*p;
	void *package;

	p = EventRspQueue[idx];
	switch(p->RspType)
	{
		case HCI_EVENT_PKG :
			package = malloc(sizeof(HCI_TimeOutEventRsp));
			if(package != NULL)
			{
				memcpy(package,(u8 *)&HCI_TimeOutEventRsp,sizeof(HCI_TimeOutEventRsp));
				*((u16 *)(((tHCI_EVENT_PKG *)package)->param)) = p->RspOpCode;
			}
			break;
		case BCI_EVENT_PKG :
			package = malloc(sizeof(BCI_TimeOutEventRsp));
			if(package != NULL)
			{
				memcpy(package,(u8 *)&BCI_TimeOutEventRsp,sizeof(BCI_TimeOutEventRsp));
				((tBCI_PACKAGE *)package)->flag |= p->RspFlag;
				((tBCI_PACKAGE *)package)->opcode = p->RspOpCode;
			}
			break;
	}
	return package;
}
/*------------------------------------------------------------------------------
* Function Name : BLE_StatusEventParser
* Description   :
* Input         :
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void BLE_StatusEventParser(tBCI_PACKAGE *pack)
{
	if((pack->type == BCI_EVENT_PKG) && (pack->opcode == BCI_STATUS))
	{
		if(((tSTATUS *)(pack->param))->b.Connect) BLE_Operate.ConnectState = TRUE;
		else	BLE_Operate.ConnectState = FALSE;
	}
}
/*------------------------------------------------------------------------------
* Function Name : BC7701_TransmitPackage
* Description   : send package data to BC7701
* Input         :
*				- length : package length
*					- If it is an HCI & BCI package, use the length in the package
*				- pbuf : package index
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
bool BLE_EventResponseEmpty(void)
{
	return(IS_ERQUEUE_EMPTY() ? TRUE:FALSE);
}
/*------------------------------------------------------------------------------
* Function Name : BLE process
* Description   :
* Input         :
* Output        : None
* Return        : None
------------------------------------------------------------------------------*/
void *BLE_Process(void)
{
	static u8	modifystep;
	u8		idx;
	void	*TxPack;

	uRspEventIdx = NULL;
	BC7701_ReceiveParserPackage();
	uReceiveIdx = BC7701_ReadReceivePackage();
	if(uReceiveIdx != NULL)
	{
		uRspEventIdx = BLE_EventResponseCheck(uReceiveIdx);
	}
	else
	{
		if(!IS_ERQUEUE_EMPTY())
		{
			idx = uERQueueRd;
			while(idx != uERQueueWr)
			{
				if(EventRspQueue[idx]->flag == EVENT_RSP_TMOUT)
				{
					uReceiveIdx = BLE_ErrorEventRsp(idx);
					if(uReceiveIdx != NULL)
					{
						uRspEventIdx = EventRspQueue[idx];
						BLE_EventRspQueueRemove(idx);
					}
					idx = uERQueueWr;
				}
				else
				{
					if(++idx >= EVTRSP_QUEUE_MAX) idx = 0;
				}
			}
		}
	}

	switch(BLE_Operate.States)
	{
		case _BLE_POWER_DOWN_ :
			break;
		case	_BLE_START_ :
#if (_FIXED_BAUD_RATE_ENABLE_	== 1)
			BC7701_InterfaceConfigure(_FIXED_BAUDRATE_);
#else
			BC7701_InterfaceConfigure(_UART_BAUDRATE_);
#endif
		case _BLE_POWER_UP_ :
			BC7701_RESET_CLR();
			BLE_Operate.TimeCounter = RESET_DELAY_TIME;
			BLE_Operate.States =	_BLE_RESET_DELAY_;
			break;
		case _BLE_RESET_DELAY_:
			if(BLE_Operate.TimeCounter == 0)
			{
				BC7701_RESET_SET();
#if (_FIXED_BAUD_RATE_ENABLE_	== 1)
				BC7701_HardwareBaudRateDefault(BAUD_RATE_115200);		//for default baud rate mode
#else
				BC7701_HardwareBaudRateDefault(_UART_BAUDRATE_);
#endif
				BLE_Operate.TimeCounter = PWRUP_DELAY_TIME;
				BLE_Operate.States =	_BLE_PWRUP_DELAY_;
			}
			break;
		case _BLE_PWRUP_DELAY_ :
			if(BLE_Operate.TimeCounter == 0)
			{
				BC7701_HardwareBaudRateRelease();
				BLE_Operate.TimeCounter = BOOT_READY_TIME;
				BLE_Operate.States = _BLE_BOOT_READY_;
			}
			break;
		case _BLE_BOOT_READY_ :
			if(BLE_Operate.TimeCounter == 0)
			{
				BLE_Operate.States = _BLE_PWRUP_EXAMINE_;
			}
			break;
		case _BLE_PWRUP_EXAMINE_:
			//get BC7701 version date code
			if((TxPack=BC7701_SendBCIReadPackage(BCI_VERSION,0x80)) != NULL)
			{
				BLE_TransceiveToEventRsp(TxPack,EVENT_RETURN_TIME,_BLE_RECORD_DATECODE_);
			}
			break;
		case _BLE_RECORD_DATECODE_:
			//get BC7701 BT device address
			if((TxPack=BC7701_SendBCIReadPackage(BCI_DEV_ADDRESS,0x00)) != NULL)
			{
				BLE_TransceiveToEventRsp(TxPack,EVENT_RETURN_TIME,_BLE_RECORD_BDADDR_);
			}
			break;
		case _BLE_RECORD_BDADDR_:
			BLE_Operate.States = _BLE_CHG_BAUD_RATE_;
			break;
		case _BLE_CHG_BAUD_RATE_:
#if	((_CHG_BAUD_RATE_ENABLE_ == 1) && (_FIXED_BAUD_RATE_ENABLE_	== 0))
			if((TxPack=BC7701_SetBaudRate(_CHANGE_BAUDRATE_)) != NULL)
			{
				BLE_TransceiveToEventRsp(TxPack,EVENT_RETURN_TIME,_BLE_CHG_UART_BAUD_);
				modifystep = 1;
			}
#else
			modifystep = 0;
			BLE_Operate.States = _BLE_PARAMETER_MODIFY;
#endif
			break;
		case _BLE_CHG_UART_BAUD_:
			if(modifystep == 1)
			{
				BLE_Operate.TimeCounter = CHG_BAUD_DELAY_TIME;
				modifystep = 0;
			}
			if((modifystep == 0) && (BLE_Operate.TimeCounter == 0))
			{
				BC7701_UARTConfigure(_CHANGE_BAUDRATE_);
				BLE_Operate.States = _BLE_PARAMETER_MODIFY;
			}
			break;
		case _BLE_PARAMETER_MODIFY:
			if((TxPack=BLE_ParameterConfigure(&modifystep)) != NULL)
			{
				BLE_TransceiveToEventRsp(TxPack,EVENT_RETURN_TIME,_BLE_PARAMETER_MODIFY);
			}
			else
			{
				if(modifystep == 0xFF)
				{
					modifystep = 0;
					BLE_Operate.States = _BLE_FEATURE_SETUP_;
				}
			}
			break;
		case _BLE_FEATURE_SETUP_:
#if (_EXTERNAL32_CFG_ENABLE_ == 1)
			if((TxPack = BC7701_SetupFeatureFlag(FEATURE_SET,(FeatureFlag)(FEATURE_STATUS_EVENT + FEATURE_EXTERNAL32K))) != NULL)
#else
			if((TxPack = BC7701_SetupFeatureFlag(FEATURE_SET,FEATURE_STATUS_EVENT)) != NULL)
#endif
			{
				BLE_TransceiveToEventRsp(TxPack,EVENT_RETURN_TIME,_BLE_ADV_ENABLE_);
			}
			break;
		case _BLE_ADV_ENABLE_:
			if((TxPack = BC7701_AdvertisingControl(ENABLE)) != NULL)
			{
				BLE_TransceiveToEventRsp(TxPack,EVENT_RETURN_TIME,_BLE_ADV_DELAY_S_);
			}
			break;
		case _BLE_ADV_DELAY_S_:
			BLE_Operate.TimeCounter = RF_CALIBRAT_TIME;
			BLE_Operate.States = _BLE_ADV_DELAY_E_;
			break;
		case _BLE_ADV_DELAY_E_:
			if(uReceiveIdx != NULL)
			{
				if((((tBCI_PACKAGE *)uReceiveIdx)->opcode == BCI_STATUS) && (((tBCI_PACKAGE *)uReceiveIdx)->param[0] & 0x10))
				{
					BLE_Operate.TimeCounter = 0;
				}
				free(uReceiveIdx);
				uReceiveIdx = NULL;
			}
			if(BLE_Operate.TimeCounter == 0)
			{
#if (_SAVE_SW_PARA_ENABLE_ == 1)
				if((TxPack=BC7701_SetupFeatureFlag(FEATURE_SET,FEATURE_PARAM_UPDATE)) != NULL)
				{
					BLE_TransceiveToEventRsp(TxPack,EVENT_RETURN_TIME,_BLE_PWR_ON_FINISH_);
				}
#else
//				BLE_Operate.States = _BLE_PWR_ON_FINISH_;
					if((TxPack=BC7701_SetupFeatureFlag(FEATURE_SET,FEATURE_PARAM_ERASE)) != NULL)
				{
					BLE_TransceiveToEventRsp(TxPack,EVENT_RETURN_TIME,_BLE_PWR_ON_FINISH_);
				}
#endif
			}
			break;
		case _BLE_EVENT_RESPONSE_ :
			if(uReceiveIdx != NULL)
			{
				if(uRspEventIdx != NULL)
				{
					if(uRspEventIdx->flag == EVENT_RSP_DONE)
					{
						switch(NextStepStatus)
						{
							case _BLE_RECORD_DATECODE_:
								memcpy(BLE_Operate.DateCode,((tBCI_PACKAGE *)uReceiveIdx)->param,6);
								break;
							case _BLE_RECORD_BDADDR_:
								memcpy(&(BLE_Operate.BDAddr.adr),((tBCI_PACKAGE *)uReceiveIdx)->param,6);
								BLE_Operate.BDAddr.type = ((tDEV_ADDR *)(((tBCI_PACKAGE *)uReceiveIdx)->param))->type;
								break;
						}
						BLE_Operate.States = NextStepStatus;
					}
					else	BLE_Operate.States = _BLE_PWR_ON_ERROR_;
					free(uRspEventIdx);
				}
				free(uReceiveIdx);
				uReceiveIdx = NULL;
			}
			break;
		case _BLE_PWR_ON_FINISH_:
			BLE_Operate.States = _BLE_PWR_ON_SUCCESS_;
			break;
		case _BLE_PWR_ON_SUCCESS_:
			if(uReceiveIdx != NULL)
			{
				if(uRspEventIdx != NULL)	free(uRspEventIdx);
				BLE_StatusEventParser(uReceiveIdx);
			}
			break;
		case _BLE_PWR_ON_ERROR_	:
			if(uReceiveIdx != NULL)
			{
				free(uReceiveIdx);
				uReceiveIdx = NULL;
			}
			break;
	}
	return(uReceiveIdx);
}

